<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

include("include/baglan.php"); // Veritabanı bağlantısı

// Sabitler
define('BASE_URL', 'https://www.megasgroup.az/');
$sitemapFile = __DIR__ . '/../sitemap.xml';

/**
 * Verilen metinden URL dostu bir slug (kısa isim) oluşturur.
 * Çeşitli karakter setlerini destekler (Türkçe, Azerice, Rusça).
 *
 * @param string $text Giriş metni.
 * @return string Oluşturulan slug.
 */
function mb_url_slug($text)
{
    $text = trim($text);
    $text = mb_strtolower($text, 'UTF-8');
    $find = array(
        // Türkçe karakterler
        'Ç', 'Ş', 'Ğ', 'Ü', 'İ', 'Ö', 'ç', 'ş', 'ğ', 'ü', 'ö', 'ı',
        // Azerice karakterler
        'Ə', 'ə', 'X', 'x', 'Q', 'q', 'Ü', 'ü', 'Ö', 'ö', 'Ç', 'ç', 'Ş', 'ş', 'Ğ', 'ğ', 'İ', 'i',
        // Rusça karakterler
        'А', 'Б', 'В', 'Г', 'Д', 'Е', 'Ё', 'Ж', 'З', 'И', 'Й', 'К', 'Л', 'М', 'Н', 'О', 'П', 'Р', 'С', 'Т', 'У', 'Ф', 'Х', 'Ц', 'Ч', 'Ш', 'Щ', 'Ъ', 'Ы', 'Ь', 'Э', 'Ю', 'Я',
        'а', 'б', 'в', 'г', 'д', 'е', 'ё', 'ж', 'з', 'и', 'й', 'к', 'л', 'м', 'н', 'о', 'п', 'р', 'с', 'т', 'у', 'ф', 'х', 'ц', 'ч', 'ш', 'щ', 'ъ', 'ы', 'ь', 'э', 'ю', 'я',
        // Diğer özel karakterler
        '+', '#', '.', '@', '&',
    );
    $replace = array(
        // Türkçe karşılıklar
        'C', 'S', 'G', 'U', 'I', 'O', 'c', 's', 'g', 'u', 'o', 'i',
        // Azerice karşılıklar
        'E', 'e', 'X', 'x', 'Q', 'q', 'U', 'u', 'O', 'o', 'C', 'c', 'S', 's', 'G', 'g', 'I', 'i',
        // Rusça karşılıklar (yaygın transliterasyon)
        'a', 'b', 'v', 'g', 'd', 'ye', 'yo', 'zh', 'z', 'i', 'y', 'k', 'l', 'm', 'n', 'o', 'p', 'r', 's', 't', 'u', 'f', 'kh', 'ts', 'ch', 'sh', 'shch', '', 'y', '', 'e', 'yu', 'ya',
        'a', 'b', 'v', 'g', 'd', 'ye', 'yo', 'zh', 'z', 'i', 'y', 'k', 'l', 'm', 'n', 'o', 'p', 'r', 's', 't', 'u', 'f', 'kh', 'ts', 'ch', 'sh', 'shch', '', 'y', '', 'e', 'yu', 'ya',
        // Diğer özel karakterler
        'plus', 'sharp', 'dot', 'at', 'and',
    );
    $text = str_replace($find, $replace, $text);
    $text = preg_replace('/[^a-z0-9-]+/', ' ', $text); // Harf ve rakam dışındaki karakterleri boşlukla değiştir
    $text = trim(preg_replace('/\s+/', ' ', $text)); // Birden fazla boşluğu tek boşluğa indir
    $text = str_replace(' ', '-', $text); // Boşlukları tire ile değiştir
    $text = trim($text, '-'); // Baştaki ve sondaki tireleri kaldır
    return $text;
}

/**
 * Bir veritabanı öğesinden (satırından) verilen tarih alanları arasından en son güncelleme tarihini alır.
 *
 * @param array $item Veritabanı satırı (dizi).
 * @param array $dateFields Olası tarih alanı isimlerinin dizisi.
 * @return string Y-m-d formatında en son tarih, geçerli bir tarih bulunamazsa mevcut tarih.
 */
function getLastModDate($item, $dateFields)
{
    $latestTimestamp = 0;
    foreach ($dateFields as $field) {
        if (isset($item[$field]) && !empty($item[$field])) {
            $timestamp = strtotime($item[$field]);
            if ($timestamp !== false && $timestamp > $latestTimestamp) {
                $latestTimestamp = $timestamp;
            }
        }
    }
    return $latestTimestamp > 0 ? date('Y-m-d', $latestTimestamp) : date('Y-m-d');
}

// Sitemap başlangıcı
$sitemap = '<?xml version="1.0" encoding="UTF-8"?>' . PHP_EOL;
$sitemap .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9" ';
$sitemap .= 'xmlns:image="http://www.google.com/schemas/sitemap-image/1.1">' . PHP_EOL;

// Statik sayfalar
$staticPages = [
    ['loc' => BASE_URL, 'priority' => 1.0, 'changefreq' => 'daily'],
    ['loc' => BASE_URL . 'haqqimizda', 'priority' => 0.8, 'changefreq' => 'monthly'],
    ['loc' => BASE_URL . 'elaqe', 'priority' => 0.8, 'changefreq' => 'monthly'],
    ['loc' => BASE_URL . 'xidmetler', 'priority' => 0.8, 'changefreq' => 'monthly'],
    ['loc' => BASE_URL . 'xeberler', 'priority' => 0.8, 'changefreq' => 'monthly'],
    ['loc' => BASE_URL . 'layiheler', 'priority' => 0.8, 'changefreq' => 'monthly'],
    ['loc' => BASE_URL . 'sertifikatlar', 'priority' => 0.8, 'changefreq' => 'monthly'],
    ['loc' => BASE_URL . 'mehsullar', 'priority' => 0.8, 'changefreq' => 'monthly'],
    ['loc' => BASE_URL . 'komandamiz', 'priority' => 0.8, 'changefreq' => 'monthly'],
    ['loc' => BASE_URL . 'cvs', 'priority' => 0.8, 'changefreq' => 'monthly'],
    ['loc' => BASE_URL . 'galeri', 'priority' => 0.8, 'changefreq' => 'monthly'],
    ['loc' => BASE_URL . 'mehsul_kateqoriya', 'priority' => 0.8, 'changefreq' => 'monthly'],
    ['loc' => BASE_URL . 'layihe_kateqoriya', 'priority' => 0.8, 'changefreq' => 'monthly'],
    ['loc' => BASE_URL . 'xidmet_kateqoriya', 'priority' => 0.8, 'changefreq' => 'monthly'],
    ['loc' => BASE_URL . 'sehifeler', 'priority' => 0.8, 'changefreq' => 'monthly'],
    ['loc' => BASE_URL . 'reyler', 'priority' => 0.8, 'changefreq' => 'monthly'],
    ['loc' => BASE_URL . 'paketler', 'priority' => 0.8, 'changefreq' => 'monthly'],
    ['loc' => BASE_URL . 'slider', 'priority' => 0.8, 'changefreq' => 'monthly'],
];

foreach ($staticPages as $page) {
    $sitemap .= "  <url>\n";
    $sitemap .= "    <loc>" . htmlspecialchars($page['loc']) . "</loc>\n";
    $sitemap .= "    <lastmod>" . date('Y-m-d') . "</lastmod>\n"; // Statik sayfalar için mevcut tarihi kullan
    $sitemap .= "    <changefreq>" . $page['changefreq'] . "</changefreq>\n";
    $sitemap .= "    <priority>" . $page['priority'] . "</priority>\n";
    $sitemap .= "  </url>\n";
}

// Dinamik içerikler için tablo yapılandırması
$tables = [
    'komandamiz' => [
        'url_prefix' => 'komandamiz',
        'date_fields' => ['guncelleme_tarihi', 'eklenme_tarihi'],
        'title_fields' => ['adi', 'onaciklama', 'aciklama'],
        'seo_field' => 'seo',
        'image_fields' => ['resim', 'icon', 'sehifesekli'], // Tutarlılık için image_fields olarak değiştirildi
        'priority' => 0.7,
        'changefreq' => 'weekly'
    ],
    'haqqimizda' => [
        'url_prefix' => 'haqqimizda',
        'date_fields' => ['guncelleme_tarihi', 'eklenme_tarihi'],
        'title_fields' => ['adi', 'onaciklama', 'aciklama'],
        'seo_field' => 'seo',
        'image_fields' => ['resim', 'icon', 'sehifesekli'], // Tutarlılık için image_fields olarak değiştirildi
        'priority' => 0.7,
        'changefreq' => 'weekly'
    ],
    'xidmetler' => [
        'url_prefix' => 'xidmetler',
        'date_fields' => ['guncelleme_tarihi', 'eklenme_tarihi'],
        'title_fields' => ['adi', 'onaciklama', 'aciklama'],
        'seo_field' => 'seo',
        'image_fields' => ['resim', 'icon', 'sehifesekli'],
        'priority' => 0.9,
        'changefreq' => 'monthly'
    ],
    'galeri' => [
        'url_prefix' => 'galeri',
        'date_fields' => ['guncelleme_tarihi', 'xidmet_kateqoriya', 'xeber_kateqoriya', 'layihe_kateqoriya', 'mehsul_kateqoriya', 'eklenme_tarihi'],
        'title_fields' => ['adi', 'onaciklama', 'aciklama'],
        'seo_field' => 'seo',
        'image_fields' => ['resim', 'resim1', 'resim2', 'resim3', 'resim4'],
        'priority' => 0.8,
        'changefreq' => 'daily'
    ],
    'mehsul_kateqoriya' => [
        'url_prefix' => 'mehsul_kateqoriya',
        'date_fields' => ['guncelleme_tarihi', 'layihe_kateqoriya', 'mehsul_kateqoriya', 'eklenme_tarihi'],
        'title_fields' => ['adi', 'onaciklama', 'aciklama'],
        'seo_field' => 'seo',
        'image_fields' => ['resim'],
        'priority' => 0.8,
        'changefreq' => 'daily'
    ],
    'layihe_kateqoriya' => [
        'url_prefix' => 'layihe_kateqoriya',
        'date_fields' => ['guncelleme_tarihi', 'eklenme_tarihi'],
        'title_fields' => ['adi', 'onaciklama', 'aciklama'],
        'seo_field' => 'seo',
        'image_fields' => ['resim'],
        'priority' => 0.8,
        'changefreq' => 'daily'
    ],
    'xidmet_kateqoriya' => [
        'url_prefix' => 'xidmet_kateqoriya',
        'date_fields' => ['guncelleme_tarihi', 'eklenme_tarihi'],
        'title_fields' => ['adi', 'onaciklama', 'aciklama'],
        'seo_field' => 'seo',
        'image_fields' => ['resim', 'resim2', 'resim3'],
        'priority' => 0.8,
        'changefreq' => 'daily'
    ],
    'xeberler' => [
        'url_prefix' => 'xeberler',
        'date_fields' => ['guncelleme_tarihi', 'eklenme_tarihi'],
        'title_fields' => ['adi', 'onaciklama', 'aciklama'],
        'seo_field' => 'seo',
        'image_fields' => ['resim', 'sehifesekli'],
        'priority' => 0.8,
        'changefreq' => 'daily'
    ],
    'layiheler' => [
        'url_prefix' => 'layiheler',
        'date_fields' => ['guncelleme_tarihi', 'elaveolunma_tarixi', 'bitme_tarixi'],
        'title_fields' => ['adi', 'onaciklama', 'aciklama'],
        'seo_field' => 'seo',
        'image_fields' => ['resim', 'sehifesekli'],
        'priority' => 0.7,
        'changefreq' => 'monthly'
    ],
    'sertifikatlar' => [
        'url_prefix' => 'sertifikatlar',
        'date_fields' => ['guncelleme_tarihi', 'elaveolunma_tarixi', 'bitme_tarixi'],
        'title_fields' => ['adi', 'onaciklama', 'aciklama'],
        'seo_field' => 'seo',
        'image_fields' => ['resim', 'sehifesekli'],
        'priority' => 0.7,
        'changefreq' => 'monthly'
    ],
    'cvs' => [
        'url_prefix' => 'cvs',
        'date_fields' => ['guncelleme_tarihi', 'elaveolunma_tarixi', 'bitme_tarixi'],
        'title_fields' => ['adi', 'aciklama'],
        'seo_field' => 'seo',
        'image_fields' => ['resim', 'resim2'],
        'priority' => 0.7,
        'changefreq' => 'monthly'
    ],
    'sehifeler' => [
        'url_prefix' => 'sehifeler',
        'date_fields' => ['guncelleme_tarihi', 'eklenme_tarihi'],
        'title_fields' => ['adi', 'onaciklama', 'aciklama'],
        'seo_field' => 'seo',
        'image_fields' => ['resim', 'icon', 'resim2'],
        'priority' => 0.7,
        'changefreq' => 'monthly'
    ],
    'reyler' => [
        'url_prefix' => 'reyler',
        'date_fields' => ['guncelleme_tarihi', 'eklenme_tarihi'],
        'title_fields' => ['adi', 'onaciklama', 'aciklama'],
        'seo_field' => 'seo',
        'image_fields' => ['resim', 'icon', 'sehifesekli', 'resim2'],
        'priority' => 0.7,
        'changefreq' => 'monthly'
    ],
    'mehsullar' => [
        'url_prefix' => 'mehsullar',
        'date_fields' => ['guncelleme_tarihi', 'eklenme_tarihi'],
        'title_fields' => ['adi', 'onaciklama', 'fiyat', 'aciklama'],
        'seo_field' => 'seo',
        'image_fields' => ['resim', 'icon', 'sehifesekli', 'resim2'],
        'priority' => 0.8,
        'changefreq' => 'weekly'
    ],
    'paketler' => [
        'url_prefix' => 'paketler',
        'date_fields' => ['guncelleme_tarihi', 'eklenme_tarihi'],
        'title_fields' => ['adi', 'onaciklama', 'fiyat', 'aciklama'],
        'seo_field' => 'seo',
        'image_fields' => ['resim', 'icon', 'sehifesekli', 'resim2'],
        'priority' => 0.8,
        'changefreq' => 'weekly'
    ],
    'referanslar' => [
        'url_prefix' => 'referanslar',
        'date_fields' => ['guncelleme_tarihi'], // Tutarlılık için date_fields olarak değiştirildi
        'title_fields' => ['adi', 'onaciklama', 'aciklama'],
        'seo_field' => 'seo',
        'image_fields' => ['resim'], // Tutarlılık için image_fields olarak değiştirildi
        'priority' => 0.8,
        'changefreq' => 'weekly'
    ],
    'slider' => [
        'url_prefix' => 'slider',
        'date_fields' => ['guncelleme_tarihi', 'eklenme_tarihi'],
        'title_fields' => ['adi', 'onaciklama', 'aciklama'],
        'seo_field' => 'seo',
        'image_fields' => ['resim', 'icon', 'resim2'],
        'priority' => 0.8,
        'changefreq' => 'weekly'
    ],
];

// Dinamik içerikleri sitemape ekle
foreach ($tables as $tableName => $config) {
    try {
        $query = "SELECT * FROM {$tableName} WHERE veziyyeti='0'"; // 'veziyyeti' alanı '0' olan kayıtları seç
        $stmt = $db->prepare($query);
        $stmt->execute();
        $items = $stmt->fetchAll(PDO::FETCH_ASSOC);

        foreach ($items as $item) {
            // URL slug'ını belirle
            $urlSlug = '';
            if (!empty($item[$config['seo_field']])) {
                $urlSlug = $item[$config['seo_field']];
            } else {
                // Slug oluşturmak için ilk uygun başlık alanını dene
                foreach ($config['title_fields'] as $titleField) {
                    if (isset($item[$titleField]) && !empty($item[$titleField])) {
                        $urlSlug = $config['url_prefix'] . '/' . mb_url_slug($item[$titleField]);
                        break; // Bulunan ilk başlık alanını kullan ve döngüyü sonlandır
                    }
                }
            }
            if (empty($urlSlug)) {
                // SEO alanı veya uygun başlık alanı yoksa yedek olarak 'id' kullan
                // 'id' alanının her tabloda mevcut olduğu varsayılır.
                $urlSlug = $config['url_prefix'] . '/' . $item['id'];
            }

            $loc = BASE_URL . $urlSlug;

            $sitemap .= "  <url>\n";
            $sitemap .= "    <loc>" . htmlspecialchars($loc) . "</loc>\n";
            $sitemap .= "    <lastmod>" . getLastModDate($item, $config['date_fields']) . "</lastmod>\n";
            $sitemap .= "    <changefreq>" . $config['changefreq'] . "</changefreq>\n";
            $sitemap .= "    <priority>" . $config['priority'] . "</priority>\n";

            // Resimleri ekle (eğer image_fields tanımlı ve boş değilse)
            if (isset($config['image_fields']) && is_array($config['image_fields'])) {
                foreach ($config['image_fields'] as $imageField) {
                    if (isset($item[$imageField]) && !empty($item[$imageField])) {
                        // Resimlerin 'imgupload' dizininde ve WebP formatında olduğu varsayılır.
                        $imageUrl = BASE_URL . 'imgupload/' . pathinfo($item[$imageField], PATHINFO_FILENAME) . '.webp';
                        
                        // Resim için uygun bir başlık bulmaya çalış
                        $imageTitle = '';
                        foreach ($config['title_fields'] as $titleField) {
                            if (isset($item[$titleField]) && !empty($item[$titleField])) {
                                $imageTitle = $item[$titleField];
                                break;
                            }
                        }

                        $sitemap .= "    <image:image>\n";
                        $sitemap .= "      <image:loc>" . htmlspecialchars($imageUrl) . "</image:loc>\n";
                        if (!empty($imageTitle)) {
                            $sitemap .= "      <image:title>" . htmlspecialchars($imageTitle) . "</image:title>\n";
                        }
                        $sitemap .= "    </image:image>\n";
                    }
                }
            }

            $sitemap .= "  </url>\n";
        }
    } catch (Exception $e) {
        error_log("Sitemap Hatası ({$tableName}): " . $e->getMessage());
    }
}

$sitemap .= '</urlset>';

// Sitemap dosyasını yaz
$result = file_put_contents($sitemapFile, $sitemap);

if ($result === false) {
    echo "Sitemap dosyası oluşturulamadı.";
} else {
    echo "Sitemap başarıyla oluşturuldu. Bayt sayısı: " . $result;
}

exit;
?>